<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

  <xsl:output method="html" encoding="UTF-8" indent="yes"
              doctype-public="-//W3C//DTD HTML 4.01//EN"/>

  <!-- ═══════════════════════════════════════════════════════════
       lyric-meter.xsl
       Transforms lyric-meter XML (from lyric_meter_analyzer.html)
       into a musician-readable HTML reference sheet.

       Usage (Saxon-JS or Saxon-HE):
         java -jar saxon-he.jar -s:mysong.xml -xsl:lyric-meter.xsl -o:mysong.html

       Or open directly in any browser that supports local XSLT
       by adding this PI to the XML file:
         <?xml-stylesheet type="text/xsl" href="lyric-meter.xsl"?>
       ═══════════════════════════════════════════════════════════ -->


  <!-- ─── ROOT ─────────────────────────────────────────────── -->
  <xsl:template match="/">
    <html lang="en">
      <head>
        <meta charset="UTF-8"/>
        <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
        <title><xsl:value-of select="song/metadata/title"/> — Lyric Meter Sheet</title>
        <link href="https://fonts.googleapis.com/css2?family=Playfair+Display:ital,wght@0,400;0,700;1,400&amp;family=Courier+Prime:ital,wght@0,400;0,700;1,400&amp;family=Jost:wght@300;400;500&amp;display=swap" rel="stylesheet"/>
        <style>
          /* ── Variables ───────────────────────────────────── */
          :root {
            --bg:       #1a1814;
            --surface:  #221f1b;
            --border:   #3a342c;
            --gold:     #c9a84c;
            --gold-lt:  #e8cc80;
            --amber:    #d4793a;
            --text:     #e8e0d4;
            --text-dim: #8a7f72;
            --stressed: #e8cc80;
            --unstress: #6a9fb5;
            --green:    #7ab87a;
            --sep:      #5a5a7a;
          }
          * { box-sizing: border-box; margin: 0; padding: 0; }

          body {
            background: var(--bg);
            color: var(--text);
            font-family: 'Jost', sans-serif;
            font-weight: 300;
            padding: 2.5rem 1.5rem 4rem;
            max-width: 860px;
            margin: 0 auto;
          }

          /* ── Print overrides ─────────────────────────────── */
          @media print {
            body { background: #fff; color: #111; max-width: 100%; padding: 1cm 2cm; }
            :root {
              --bg: #fff; --surface: #f5f5f0; --border: #ccc;
              --gold: #7a5c00; --gold-lt: #5a3c00; --amber: #8a4000;
              --text: #111; --text-dim: #555;
              --stressed: #7a3000; --unstress: #005577; --green: #2a6e2a;
              --sep: #888;
            }
            .no-print { display: none; }
            section.lyric-section { page-break-inside: avoid; }
          }

          /* ── Header ──────────────────────────────────────── */
          header {
            border-bottom: 2px solid var(--gold);
            padding-bottom: 1.2rem;
            margin-bottom: 2rem;
          }
          header h1 {
            font-family: 'Playfair Display', serif;
            font-size: clamp(1.6rem, 5vw, 2.8rem);
            color: var(--gold);
            letter-spacing: 0.03em;
            line-height: 1.15;
          }
          header h1 em { color: var(--gold-lt); font-style: italic; }
          .header-sub {
            margin-top: 0.5rem;
            font-size: 0.8rem;
            color: var(--text-dim);
            letter-spacing: 0.1em;
            text-transform: uppercase;
          }

          /* ── Meter summary card ───────────────────────────── */
          .meter-summary {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
            gap: 0.8rem;
            margin-bottom: 2.5rem;
          }
          .meter-card {
            background: var(--surface);
            border: 1px solid var(--border);
            border-radius: 8px;
            padding: 0.8rem 1rem;
            text-align: center;
          }
          .meter-card .mc-label {
            font-size: 0.62rem;
            letter-spacing: 0.12em;
            text-transform: uppercase;
            color: var(--text-dim);
            margin-bottom: 0.3rem;
          }
          .meter-card .mc-value {
            font-family: 'Playfair Display', serif;
            font-size: 1.2rem;
            color: var(--gold-lt);
            line-height: 1.2;
          }
          .meter-card .mc-value.small { font-size: 0.95rem; padding-top: 0.15rem; }

          /* ── Section ─────────────────────────────────────── */
          section.lyric-section {
            margin-bottom: 2.5rem;
          }
          .section-header {
            display: flex;
            align-items: baseline;
            gap: 0.8rem;
            margin-bottom: 1.2rem;
            padding-bottom: 0.5rem;
            border-bottom: 1px solid var(--border);
          }
          .section-badge {
            font-family: 'Playfair Display', serif;
            font-size: 1.5rem;
            color: var(--gold);
            line-height: 1;
            min-width: 1.6rem;
          }
          .section-name {
            font-size: 0.78rem;
            letter-spacing: 0.1em;
            text-transform: uppercase;
            color: var(--text-dim);
          }
          .section-form-tag {
            margin-left: auto;
            font-size: 0.65rem;
            letter-spacing: 0.08em;
            text-transform: uppercase;
            padding: 0.15rem 0.5rem;
            border: 1px solid var(--border);
            border-radius: 20px;
            color: var(--text-dim);
          }

          /* ── Lyric lines ─────────────────────────────────── */
          .line-block {
            margin-bottom: 1.4rem;
          }
          .line-block:last-child { margin-bottom: 0; }

          /*
            Stress marks display.
            Each syllable mark is rendered as an inline-block span,
            sized to match one character of Courier Prime so it
            sits neatly above the lyric syllable.
          */
          .stress-row {
            font-family: 'Courier Prime', monospace;
            font-size: 0.88rem;
            letter-spacing: 0.04em;
            line-height: 1.2;
            margin-bottom: 0.05rem;
            white-space: pre;
          }
          .mark-stressed {
            color: var(--stressed);
            font-weight: 700;
            font-size: 1rem;
          }
          .mark-unstressed {
            color: var(--unstress);
            font-size: 0.88rem;
          }
          .mark-sep {
            color: var(--sep);
            font-size: 0.75rem;
          }
          .mark-space { color: transparent; }

          .lyric-line {
            font-family: 'Courier Prime', monospace;
            font-size: 1.05rem;
            color: var(--text);
            line-height: 1.5;
            white-space: pre-wrap;
          }

          .line-meta {
            margin-top: 0.3rem;
            display: flex;
            gap: 0.5rem;
            align-items: center;
            flex-wrap: wrap;
          }
          .tag {
            font-size: 0.62rem;
            letter-spacing: 0.07em;
            text-transform: uppercase;
            padding: 0.15rem 0.45rem;
            border-radius: 20px;
            font-weight: 500;
          }
          .tag-foot    { background: #2e2a22; border: 1px solid var(--gold);   color: var(--gold); }
          .tag-syl     { background: #1e2a1e; border: 1px solid var(--green);  color: var(--green); }
          .tag-pattern { font-family: 'Courier Prime', monospace; font-size: 0.68rem;
                         background: #1e1e2a; border: 1px solid #5a5a8a; color: #9090c0; }

          /* ── Footer ──────────────────────────────────────── */
          footer {
            margin-top: 3rem;
            padding-top: 1rem;
            border-top: 1px solid var(--border);
            font-size: 0.7rem;
            color: var(--text-dim);
            letter-spacing: 0.06em;
            display: flex;
            justify-content: space-between;
            flex-wrap: wrap;
            gap: 0.4rem;
          }

          /* ── Legend ──────────────────────────────────────── */
          .legend {
            background: var(--surface);
            border: 1px solid var(--border);
            border-radius: 8px;
            padding: 0.8rem 1.2rem;
            margin-bottom: 2rem;
            font-size: 0.75rem;
            color: var(--text-dim);
            display: flex;
            gap: 1.5rem;
            flex-wrap: wrap;
            align-items: center;
          }
          .legend-item { display: flex; align-items: center; gap: 0.35rem; }
          .legend-mark { font-family: 'Courier Prime', monospace; font-size: 1rem; }
        </style>
      </head>
      <body>

        <!-- ── Header ───────────────────────────────────────── -->
        <header>
          <h1><xsl:value-of select="song/metadata/title"/></h1>
          <div class="header-sub">
            <xsl:value-of select="song/metadata/genre"/>
            <xsl:text> · </xsl:text>
            <xsl:value-of select="song/lyrics/@form"/> form
            <xsl:text> · </xsl:text>
            <xsl:value-of select="song/metadata/date"/>
          </div>
        </header>

        <!-- ── Meter summary ────────────────────────────────── -->
        <xsl:call-template name="meterSummary"/>

        <!-- ── Legend ───────────────────────────────────────── -->
        <div class="legend no-print">
          <div class="legend-item">
            <span class="legend-mark mark-stressed">/</span>
            <span>Stressed syllable</span>
          </div>
          <div class="legend-item">
            <span class="legend-mark mark-unstressed">u</span>
            <span>Unstressed syllable</span>
          </div>
          <div class="legend-item">
            <span class="legend-mark mark-sep">|</span>
            <span>Word boundary</span>
          </div>
        </div>

        <!-- ── Sections ─────────────────────────────────────── -->
        <xsl:apply-templates select="song/lyrics/section"/>

        <!-- ── Footer ───────────────────────────────────────── -->
        <footer>
          <span>Generated by Lyric Meter Analyzer v2.0-cmu</span>
          <span>Validated against lyric-meter.xsd</span>
        </footer>

      </body>
    </html>
  </xsl:template>


  <!-- ─── METER SUMMARY ────────────────────────────────────── -->
  <xsl:template name="meterSummary">
    <div class="meter-summary">
      <div class="meter-card">
        <div class="mc-label">Time Signature</div>
        <div class="mc-value"><xsl:value-of select="song/metadata/time-signature"/></div>
      </div>
      <div class="meter-card">
        <div class="mc-label">Form</div>
        <div class="mc-value"><xsl:value-of select="song/lyrics/@form"/></div>
      </div>
      <div class="meter-card">
        <div class="mc-label">Sections</div>
        <div class="mc-value"><xsl:value-of select="count(song/lyrics/section)"/></div>
      </div>
      <div class="meter-card">
        <div class="mc-label">Total Lines</div>
        <div class="mc-value"><xsl:value-of select="count(song/lyrics/section/line)"/></div>
      </div>
      <div class="meter-card">
        <div class="mc-label">Genre</div>
        <div class="mc-value small"><xsl:value-of select="song/metadata/genre"/></div>
      </div>
    </div>
  </xsl:template>


  <!-- ─── SECTION ──────────────────────────────────────────── -->
  <xsl:template match="section">
    <section class="lyric-section">
      <div class="section-header">
        <span class="section-badge">
          <xsl:value-of select="@label"/>
          <xsl:if test="@verseNumber != ''">
            <xsl:text>&#x00B9;</xsl:text><!-- superscript placeholder; use verse num -->
          </xsl:if>
        </span>
        <span class="section-name"><xsl:value-of select="@name"/></span>
        <xsl:if test="@verseNumber != ''">
          <span class="section-form-tag">Verse <xsl:value-of select="@verseNumber"/></span>
        </xsl:if>
      </div>

      <xsl:apply-templates select="line"/>
    </section>
  </xsl:template>


  <!-- ─── LINE ─────────────────────────────────────────────── -->
  <xsl:template match="line">
    <div class="line-block">

      <!-- Stress marks row -->
      <xsl:if test="@stress">
        <div class="stress-row">
          <xsl:call-template name="renderStress">
            <xsl:with-param name="str" select="@stress"/>
          </xsl:call-template>
        </div>
      </xsl:if>

      <!-- Lyric text — normalize whitespace from the XML indentation -->
      <div class="lyric-line">
        <xsl:value-of select="normalize-space(.)"/>
      </div>

      <!-- Tags row -->
      <div class="line-meta">
        <xsl:if test="@foot">
          <span class="tag tag-foot"><xsl:value-of select="@foot"/></span>
        </xsl:if>
        <xsl:if test="@syllables">
          <span class="tag tag-syl"><xsl:value-of select="@syllables"/> syl</span>
        </xsl:if>
        <xsl:if test="@stress">
          <span class="tag tag-pattern">
            <xsl:value-of select="@stress"/>
          </span>
        </xsl:if>
      </div>

    </div>
  </xsl:template>


  <!-- ─── STRESS STRING RENDERER ───────────────────────────── -->
  <!--
    Walks the @stress string character by character using XSLT 1.0
    substring() recursion. Each character is wrapped in a styled span:
      /  →  mark-stressed
      u  →  mark-unstressed
      |  →  mark-sep
      (space) → mark-space (preserves alignment)
  -->
  <xsl:template name="renderStress">
    <xsl:param name="str"/>
    <xsl:if test="string-length($str) > 0">
      <xsl:variable name="ch" select="substring($str, 1, 1)"/>
      <xsl:choose>
        <xsl:when test="$ch = '/'">
          <span class="mark-stressed">/</span>
        </xsl:when>
        <xsl:when test="$ch = 'u'">
          <span class="mark-unstressed">u</span>
        </xsl:when>
        <xsl:when test="$ch = '|'">
          <span class="mark-sep"> | </span>
        </xsl:when>
        <xsl:otherwise>
          <!-- space — emit a non-breaking space to preserve layout -->
          <xsl:text> </xsl:text>
        </xsl:otherwise>
      </xsl:choose>
      <!-- Recurse on the remainder -->
      <xsl:call-template name="renderStress">
        <xsl:with-param name="str" select="substring($str, 2)"/>
      </xsl:call-template>
    </xsl:if>
  </xsl:template>

</xsl:stylesheet>
